<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PersonaResource\Pages;
use App\Filament\Resources\PersonaResource\RelationManagers;
use App\Models\Persona;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Components\Fieldset;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\FileUpload;
use Illuminate\Support\Facades\Storage;
use Filament\Forms\Components\Toggle;
use Filament\Tables\Columns\IconColumn;
use Illuminate\Support\HtmlString;
use Filament\Tables\Columns\ViewColumn;

class PersonaResource extends Resource
{
    protected static ?string $model = Persona::class;

    protected static ?int $navigationSort = 1;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Fieldset::make('Avanzado')
                    ->schema([
                        Toggle::make('alerta_sofia')
                            ->label('¿Es Alerta Sofía?')
                            ->required(),
                        Toggle::make('missing_children')
                            ->label('¿Missing Children?')
                            ->required(),
                        Toggle::make('top')
                            ->label('¿Ubicar en Máxima Difusión? 5 publicaciones máximo')
                            ->required(),
                    ])
                    ->columns(3),
                Fieldset::make('Información')
                    ->schema([
                        Forms\Components\TextInput::make('nombre')
                            ->label('Nombre Completo')
                            ->required(),
                        Forms\Components\DatePicker::make('fecha_nacimiento')
                            ->label('Fecha de Nacimiento')
                    ])
                    ->columns(2),
                Fieldset::make('Ubicación')
                    ->schema([
                        Forms\Components\TextInput::make('ubicacion')
                            ->label('Lugar de Desaparición'),
                        Forms\Components\DatePicker::make('fecha_desaparicion')
                            ->label('Fecha de Desaparición')
                    ])
                    ->columns(2),
                Fieldset::make('Tipo')
                    ->schema([
                        Forms\Components\Select::make('categoria_id')
                            ->required()
                            ->relationship(name: 'categoria', titleAttribute: 'descripcion'),
                        Forms\Components\TextInput::make('recompensa')
                            ->label('Recompensa')
                            ->placeholder('$ 2.000.000')
                    ])
                    ->columns(2),
                Fieldset::make('Imágen')
                    ->schema([
                        FileUpload::make('imagen')
                            ->label('Imágen Principal')
                            ->image()
                            ->imageResizeMode('contain')
                            ->imageEditorEmptyFillColor('#32bced')
                            ->imageResizeTargetWidth('507')
                            ->imageResizeTargetHeight('368')
                            ->deleteUploadedFileUsing(function ($file) {
                                Storage::disk('public')->delete($file);
                            })
                    ])
                    ->columns(1),
                Fieldset::make('Origen')
                    ->schema([
                        Forms\Components\TextInput::make('link')
                            ->prefix('https://')
                    ])
                    ->columns(1),
                Fieldset::make('Información Adicional')
                    ->schema([
                        Forms\Components\RichEditor::make('informacion')
                            ->label('Información')
                            ->disableToolbarButtons([
                                'attachFiles',
                                'orderedList',
                                'bulletList',
                                'codeBlock',
                            ])
                    ])
                    ->columns(1),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                IconColumn::make('top')
                    ->label('Prioridad')
                    ->alignCenter()
                    ->boolean()
                    ->trueColor('success'),
                ViewColumn::make('asofia')->label('')->view('filament.tables.columns.alerta-sofia'),
                ViewColumn::make('mchild')->label('')->view('filament.tables.columns.missing-children'),
                Tables\Columns\ImageColumn::make('imagen')
                    ->alignCenter()
                    ->circular(),
                Tables\Columns\TextColumn::make('nombre')
                    ->label('Nombre')
                    ->searchable()
                    ->description(function (Persona $record): ?HtmlString {
                        return new HtmlString('<span style="font-size: 10px; background-color: '.$record->categoria->color.'; color: white; padding: 4px 8px; text-align: center; border-radius: 5px;">'.$record->categoria->descripcion.'</span>');
                    }),
                Tables\Columns\TextColumn::make('ubicacion')
                    ->label('Ubicación')
                    ->limit(25),
                Tables\Columns\TextColumn::make('fecha_desaparicion')
                    ->label('Desaparición')
                    ->date(),
            ])->defaultSort(function (Builder $query): Builder {
                return $query
                    ->orderBy('alerta_sofia', 'DESC')
                    ->orderBy('top', 'DESC')
                    ->orderBy('fecha_desaparicion', 'DESC');
            })
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    //Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPersonas::route('/'),
            'create' => Pages\CreatePersona::route('/create'),
            'edit' => Pages\EditPersona::route('/{record}/edit'),
        ];
    }
}
